/**
 * CLX GMaps AI Review - Popup Script
 * Settings management and UI interaction
 */

// DOM Elements
let elements = {};

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  console.log('[CLX Popup] DOM loaded, initializing...');
  
  try {
    // Get DOM Elements
    elements = {
      version: document.getElementById('version'),
      uuid: document.getElementById('uuid'),
      copyUuid: document.getElementById('copyUuid'),
      userName: document.getElementById('userName'),
      saveName: document.getElementById('saveName'),
      nameLockIcon: document.getElementById('nameLockIcon'),
      nameRequiredHint: document.getElementById('nameRequiredHint'),
      statusChip: document.getElementById('statusChip'),
      statusText: document.getElementById('statusText'),
      inactiveWarning: document.getElementById('inactiveWarning'),
      paymentRequest: document.getElementById('paymentRequest'),
      purchaseLink: document.getElementById('purchaseLink'),
      segmentOpenai: document.getElementById('segmentOpenai'),
      segmentAnthropic: document.getElementById('segmentAnthropic'),
      segmentDu: document.getElementById('segmentDu'),
      segmentSie: document.getElementById('segmentSie'),
      saveSettings: document.getElementById('saveSettings'),
      saveMessage: document.getElementById('saveMessage'),
      logCount: document.getElementById('logCount'),
      exportLogs: document.getElementById('exportLogs')
    };

    // Verify all elements exist
    for (const [key, el] of Object.entries(elements)) {
      if (!el) {
        console.error(`[CLX Popup] Element not found: ${key}`);
      }
    }

    // Load version from manifest
    try {
      const manifest = chrome.runtime.getManifest();
      elements.version.textContent = `v${manifest.version}`;
    } catch (e) {
      console.error('[CLX Popup] Failed to get manifest:', e);
    }

    // Setup event listeners FIRST (before async operations)
    setupEventListeners();
    console.log('[CLX Popup] Event listeners attached');

    // Then load data asynchronously
    loadAllData();
    
  } catch (error) {
    console.error('[CLX Popup] Initialization error:', error);
  }
});

/**
 * Setup all event listeners
 */
function setupEventListeners() {
  // Copy UUID
  if (elements.copyUuid) {
    elements.copyUuid.addEventListener('click', () => {
      console.log('[CLX Popup] Copy UUID clicked');
      const uuid = elements.uuid?.textContent;
      if (uuid && uuid !== 'Laden...' && uuid !== 'Nicht verfügbar') {
        navigator.clipboard.writeText(uuid).then(() => {
          showCopiedTooltip();
        }).catch(err => {
          console.error('[CLX Popup] Copy failed:', err);
        });
      }
    });
  }

  // Name input - Save button
  if (elements.saveName) {
    elements.saveName.addEventListener('click', handleSaveName);
  }

    // Name input - Enter key
    if (elements.userName) {
      elements.userName.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && !elements.userName.disabled) {
          handleSaveName();
        }
      });
      elements.userName.addEventListener('input', async () => {
        const nameValue = elements.userName.value.trim();
        const nameIsSet = await isNameSet();
        
        // Show save button when typing (if not locked and name not already set)
        if (!elements.userName.disabled && !nameIsSet && nameValue.length >= 2) {
          if (elements.saveName) {
            elements.saveName.style.display = 'flex';
          }
          if (elements.nameRequiredHint) {
            elements.nameRequiredHint.style.display = 'none';
          }
        } else if (nameValue.length < 2) {
          if (elements.saveName) {
            elements.saveName.style.display = 'none';
          }
        }
        
        // Show hint if name is required but not set
        if (!nameIsSet && nameValue.length < 2) {
          if (elements.nameRequiredHint) elements.nameRequiredHint.style.display = 'flex';
        } else {
          if (elements.nameRequiredHint) elements.nameRequiredHint.style.display = 'none';
        }
      });
      
      // Show hint on focus if name is not set
      elements.userName.addEventListener('focus', async () => {
        const nameIsSet = await isNameSet();
        if (!nameIsSet && elements.userName.value.trim().length < 2) {
          if (elements.nameRequiredHint) elements.nameRequiredHint.style.display = 'flex';
        }
      });
    }

  // AI Selection - ChatGPT
  if (elements.segmentOpenai) {
    elements.segmentOpenai.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] OpenAI selected');
      updateAISelection('openai');
    });
  }

  // AI Selection - Claude
  if (elements.segmentAnthropic) {
    elements.segmentAnthropic.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] Anthropic selected');
      updateAISelection('anthropic');
    });
  }

  // Formality Selection - Du
  if (elements.segmentDu) {
    elements.segmentDu.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] Du form selected');
      updateFormalitySelection('du');
    });
  }

  // Formality Selection - Sie
  if (elements.segmentSie) {
    elements.segmentSie.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] Sie form selected');
      updateFormalitySelection('sie');
    });
  }

  // Save settings
  if (elements.saveSettings) {
    elements.saveSettings.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] Save settings clicked');
      saveSettingsHandler();
    });
  }

  // Export logs
  if (elements.exportLogs) {
    elements.exportLogs.addEventListener('click', (e) => {
      e.preventDefault();
      console.log('[CLX Popup] Export logs clicked');
      exportLogsHandler();
    });
  }

  // Purchase link - ensure UUID is in URL when clicked
  if (elements.purchaseLink) {
    elements.purchaseLink.addEventListener('click', (e) => {
      // Update link right before navigation to ensure UUID is included
      updatePurchaseLink();
      const uuid = elements.uuid ? elements.uuid.textContent : null;
      if (!uuid || uuid === 'Laden...' || uuid === 'Nicht verfügbar' || uuid === 'Fehler beim Laden') {
        e.preventDefault();
        console.error('[CLX Popup] Cannot navigate: UUID not available');
        alert('UUID wird noch geladen. Bitte warten Sie einen Moment und versuchen Sie es erneut.');
        return false;
      }
      // Link will navigate normally with updated href
      console.log('[CLX Popup] Navigating to purchase page with UUID:', uuid);
    });
  }
}

/**
 * Load all data asynchronously
 */
async function loadAllData() {
  // Store screen info for fingerprinting (only once)
  try {
    const stored = await chrome.storage.local.get('clx_fingerprint_screen');
    if (!stored.clx_fingerprint_screen && typeof screen !== 'undefined') {
      const screenInfo = `${screen.width}x${screen.height}x${screen.colorDepth}`;
      await chrome.storage.local.set({ clx_fingerprint_screen: screenInfo });
    }
  } catch (e) {
    console.warn('[CLX Popup] Failed to store screen info:', e);
  }
  
  // Always load UUID first
  await loadUUID();
  
  // Load name and check if it's set
  await loadUserName();
  const nameIsSet = await isNameSet();
  
  if (nameIsSet) {
    // Name is set - load full license data and settings
    await loadLicenseData();
    await loadSettings();
    await loadLogCount();
  } else {
    // Name not set - show default status message
    showDefaultStatus();
    // Keep saveSettings button ENABLED so user can save name + settings together
    if (elements.saveSettings) {
      elements.saveSettings.disabled = false;
      elements.saveSettings.title = 'Name und Einstellungen speichern';
    }
    // Hide payment request until name is set
    if (elements.paymentRequest) {
      elements.paymentRequest.style.display = 'none';
    }
    // Load default settings
    await loadSettings();
  }
}

/**
 * Update purchase link with UUID
 */
function updatePurchaseLink() {
  if (!elements.purchaseLink) return;
  
  const uuid = elements.uuid ? elements.uuid.textContent : null;
  if (uuid && uuid !== 'Laden...' && uuid !== 'Nicht verfügbar' && uuid !== 'Fehler beim Laden') {
    elements.purchaseLink.href = `https://clx-gmaps-ai-review.celox.io/#pricing?uuid=${uuid}`;
    console.log('[CLX Popup] Purchase link updated with UUID:', uuid);
  }
}

/**
 * Load UUID only
 */
async function loadUUID() {
  try {
    console.log('[CLX Popup] Loading UUID...');
    const response = await sendMessage({ action: 'getLicenseData' });
    console.log('[CLX Popup] License response:', response);
    
    if (response && !response.error && response.uuid) {
      if (elements.uuid) {
        elements.uuid.textContent = response.uuid;
      }
      // Update purchase link whenever UUID is loaded
      updatePurchaseLink();
    } else {
      console.error('[CLX Popup] License error:', response?.error || 'No UUID in response');
      // Retry after a short delay
      setTimeout(async () => {
        const retryResponse = await sendMessage({ action: 'getLicenseData' });
        if (retryResponse && retryResponse.uuid) {
          if (elements.uuid) elements.uuid.textContent = retryResponse.uuid;
          // Update purchase link when UUID is loaded
          updatePurchaseLink();
        } else {
          if (elements.uuid) elements.uuid.textContent = 'Fehler beim Laden';
        }
      }, 500);
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to load UUID:', error);
    if (elements.uuid) elements.uuid.textContent = 'Fehler beim Laden';
  }
}

/**
 * Show default status when name is not set
 */
function showDefaultStatus() {
  if (elements.statusText) {
    elements.statusText.textContent = 'Einstellungen speichern um zu starten';
  }
  if (elements.statusChip) {
    elements.statusChip.className = 'status-chip active';
    // Show "3/3 Token frei" as additional info
    const chipText = elements.statusChip.textContent || '';
    if (!chipText.includes('3/3')) {
      // Add generation info if not already present
      const statusInfo = document.createElement('span');
      statusInfo.textContent = ' (3/3 Token frei)';
      statusInfo.style.marginLeft = '0.5rem';
      statusInfo.style.fontSize = '0.875rem';
      statusInfo.style.opacity = '0.8';
      elements.statusChip.appendChild(statusInfo);
    }
  }
}

/**
 * Load license data (only called when name is set)
 */
async function loadLicenseData() {
  try {
    console.log('[CLX Popup] Loading license data...');
    const response = await sendMessage({ action: 'getLicenseData' });
    console.log('[CLX Popup] License response:', response);
    
    if (response && !response.error && response.uuid) {
      // Update UUID in DOM element
      if (elements.uuid) {
        elements.uuid.textContent = response.uuid;
      }
      
      // Always update purchase link when UUID is available
      updatePurchaseLink();
      
      // Check if user is inactive (from server response)
      const isInactive = response.user && (response.user.is_active === false || response.user.is_active === 0);
      
      if (elements.statusText) {
        if (isInactive) {
          elements.statusText.textContent = '⚠️ Account deaktiviert';
        } else {
          elements.statusText.textContent = response.statusString || 'Unbekannt';
        }
      }
      if (elements.statusChip) {
        if (isInactive) {
          elements.statusChip.className = 'status-chip inactive';
        } else {
          elements.statusChip.className = 'status-chip ' + (response.statusType || 'expired');
        }
        // Remove any appended status info
        const statusInfo = elements.statusChip.querySelector('span');
        if (statusInfo) {
          statusInfo.remove();
        }
      }
      
      // Show inactive warning
      if (isInactive) {
        if (elements.inactiveWarning) elements.inactiveWarning.style.display = 'flex';
        if (elements.saveSettings) elements.saveSettings.disabled = true;
      } else {
        if (elements.inactiveWarning) elements.inactiveWarning.style.display = 'none';
      }
      
      // Show payment request if no generations remaining
      if (response.statusType === 'expired' || (response.needsPayment !== undefined && response.needsPayment)) {
        if (elements.paymentRequest) elements.paymentRequest.style.display = 'block';
      } else {
        if (elements.paymentRequest) elements.paymentRequest.style.display = 'none';
      }
    } else {
      console.error('[CLX Popup] License error:', response?.error || 'No UUID in response');
      // Retry after a short delay
      setTimeout(async () => {
        const retryResponse = await sendMessage({ action: 'getLicenseData' });
        if (retryResponse && retryResponse.uuid) {
          // Update UUID in DOM element
          if (elements.uuid) {
            elements.uuid.textContent = retryResponse.uuid;
          }
          // Always update purchase link when UUID is available
          updatePurchaseLink();
          
          if (elements.statusText) elements.statusText.textContent = retryResponse.statusString || 'Unbekannt';
          if (elements.statusChip) elements.statusChip.className = 'status-chip ' + (retryResponse.statusType || 'expired');
          
          // Show payment request if no generations remaining
          if (retryResponse.statusType === 'expired' && retryResponse.needsPayment !== undefined) {
            if (elements.paymentRequest) elements.paymentRequest.style.display = 'block';
          } else {
            if (elements.paymentRequest) elements.paymentRequest.style.display = 'none';
          }
        } else {
          if (elements.statusText) elements.statusText.textContent = 'Fehler';
        }
      }, 500);
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to load license:', error);
    if (elements.statusText) elements.statusText.textContent = 'Fehler';
  }
}

/**
 * Load user name
 */
async function loadUserName() {
  try {
    console.log('[CLX Popup] Loading user name...');
    const licenseData = await sendMessage({ action: 'getLicenseData' });
    
    if (!licenseData || !licenseData.uuid) {
      console.error('[CLX Popup] No license data available');
      return;
    }
    
    // Check server for user name
    try {
      const response = await fetch('https://clx-gmaps-ai-review.celox.io/license/status/' + licenseData.uuid);
      const userData = await response.json();
      
      if (userData && userData.found && userData.user && userData.user.name) {
        // Name is set on server
        if (elements.userName) {
          elements.userName.value = userData.user.name;
          elements.userName.disabled = true;
          if (elements.nameLockIcon) elements.nameLockIcon.style.display = 'block';
          if (elements.saveName) elements.saveName.style.display = 'none';
        }
        return; // Name is set, nothing more to do
      }
    } catch (error) {
      console.error('[CLX Popup] Failed to check server for name:', error);
    }
    
    // No name set - require user to enter it
    if (elements.userName) {
      elements.userName.disabled = false;
      elements.userName.required = true;
      elements.userName.focus();
      if (elements.nameLockIcon) elements.nameLockIcon.style.display = 'none';
      if (elements.saveName) {
        // Initially hide save button - will show when user types
        elements.saveName.style.display = 'none';
      }
      
      // Show hint that name is required
      if (elements.nameRequiredHint) {
        elements.nameRequiredHint.style.display = 'flex';
      }
      
      // Show warning that name is required
      showSaveMessage('Bitte geben Sie zuerst Ihren Namen ein', 'info');
      
      // Check if there's already a value in the input field
      const currentValue = elements.userName.value.trim();
      if (currentValue.length >= 2) {
        // If there's already a value, show the save button
        if (elements.saveName) {
          elements.saveName.style.display = 'flex';
        }
        if (elements.nameRequiredHint) {
          elements.nameRequiredHint.style.display = 'none';
        }
      }
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to load user name:', error);
  }
}

/**
 * Check if name is set
 */
async function isNameSet() {
  try {
    const licenseData = await sendMessage({ action: 'getLicenseData' });
    if (!licenseData || !licenseData.uuid) {
      return false;
    }
    
    const response = await fetch('https://clx-gmaps-ai-review.celox.io/license/status/' + licenseData.uuid);
    const userData = await response.json();
    
    return userData && userData.found && userData.user && userData.user.name && userData.user.name.trim().length >= 2;
  } catch (error) {
    console.error('[CLX Popup] Failed to check if name is set:', error);
    return false;
  }
}

/**
 * Handle save name
 */
async function handleSaveName() {
  if (!elements.userName) return;
  
  const name = elements.userName.value.trim();
  
  // Validate
  if (name.length < 2) {
    showSaveMessage('Name muss mindestens 2 Zeichen lang sein', 'error');
    elements.userName.focus();
    return;
  }
  
  if (name.length > 100) {
    showSaveMessage('Name darf maximal 100 Zeichen lang sein', 'error');
    elements.userName.focus();
    return;
  }
  
  // Disable input while saving
  elements.userName.disabled = true;
  if (elements.saveName) elements.saveName.style.display = 'none';
  showSaveMessage('Speichere Name...', 'info');
  
  try {
    console.log('[CLX Popup] Saving user name...', name);
    const licenseData = await sendMessage({ action: 'getLicenseData' });
    console.log('[CLX Popup] License data:', licenseData);
    
    if (!licenseData || !licenseData.uuid) {
      throw new Error('UUID nicht verfügbar');
    }
    
    console.log('[CLX Popup] Sending name to server:', { uuid: licenseData.uuid, name });
    
    const requestBody = {
      uuid: licenseData.uuid,
      name: name
    };
    
    console.log('[CLX Popup] Sending POST request to /api/user/name:', requestBody);
    
    const response = await fetch('https://clx-gmaps-ai-review.celox.io/api/user/name', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(requestBody)
    });
    
    console.log('[CLX Popup] Server response status:', response.status);
    console.log('[CLX Popup] Server response headers:', response.headers);
    
    let result;
    try {
      const responseText = await response.text();
      console.log('[CLX Popup] Server response text:', responseText);
      result = JSON.parse(responseText);
      console.log('[CLX Popup] Server response parsed:', result);
    } catch (parseError) {
      console.error('[CLX Popup] Failed to parse response:', parseError);
      throw new Error('Ungültige Antwort vom Server');
    }
    
    if (!response.ok) {
      const errorMsg = result?.error || `HTTP ${response.status}`;
      console.error('[CLX Popup] Server error:', errorMsg, result);
      throw new Error(errorMsg);
    }
    
    if (result.success) {
      // Name saved successfully, lock it
      if (elements.nameLockIcon) elements.nameLockIcon.style.display = 'block';
      if (elements.saveName) elements.saveName.style.display = 'none';
      if (elements.nameRequiredHint) elements.nameRequiredHint.style.display = 'none';
      showSaveMessage('Name gespeichert ✓', 'success');
      
      // Reload all data to get updated status (should show 3/3 Token frei)
      setTimeout(async () => {
        await loadUserName(); // Reload to show locked state
        await loadLicenseData(); // Load real status (should show 3 free generations)
        
        // Now enable settings and load them
        if (elements.saveSettings) {
          elements.saveSettings.disabled = false;
          elements.saveSettings.title = '';
        }
        await loadSettings();
        await loadLogCount();
      }, 500);
    } else {
      throw new Error(result.error || 'Fehler beim Speichern');
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to save name:', error);
    elements.userName.disabled = false;
    if (elements.saveName) elements.saveName.style.display = 'flex';
    showSaveMessage(error.message || 'Fehler beim Speichern des Namens', 'error');
    elements.userName.focus();
  }
}

/**
 * Load settings
 */
async function loadSettings() {
  try {
    console.log('[CLX Popup] Loading settings...');
    const settings = await sendMessage({ action: 'getSettings' });
    console.log('[CLX Popup] Settings response:', settings);
    
    if (settings && !settings.error) {
      // Default to 'openai' if not set
      const selectedAI = settings.selectedAI || 'openai';
      updateAISelection(selectedAI);
      updateFormalitySelection(settings.formality || 'sie');
    } else {
      console.error('[CLX Popup] Settings error:', settings?.error);
      // Set default to openai if error
      updateAISelection('openai');
      updateFormalitySelection('sie');
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to load settings:', error);
    // Set default to openai on error
    updateAISelection('openai');
    updateFormalitySelection('sie');
  }
}

/**
 * Load log count
 */
async function loadLogCount() {
  try {
    console.log('[CLX Popup] Loading log count...');
    const response = await sendMessage({ action: 'getLogs' });
    const count = response?.count || 0;
    if (elements.logCount) {
      elements.logCount.textContent = `${count} Einträge`;
    }
  } catch (error) {
    console.error('[CLX Popup] Failed to load log count:', error);
  }
}

/**
 * Update AI selection UI
 */
function updateAISelection(ai) {
  console.log('[CLX Popup] Updating AI selection:', ai);
  if (elements.segmentOpenai) {
    elements.segmentOpenai.classList.toggle('active', ai === 'openai');
  }
  if (elements.segmentAnthropic) {
    elements.segmentAnthropic.classList.toggle('active', ai === 'anthropic');
  }
}

/**
 * Get currently selected AI
 */
function getSelectedAI() {
  const isOpenAI = elements.segmentOpenai?.classList.contains('active');
  return isOpenAI ? 'openai' : 'anthropic';
}

/**
 * Update formality selection UI
 */
function updateFormalitySelection(formality) {
  console.log('[CLX Popup] Updating formality selection:', formality);
  if (elements.segmentDu) {
    elements.segmentDu.classList.toggle('active', formality === 'du');
  }
  if (elements.segmentSie) {
    elements.segmentSie.classList.toggle('active', formality === 'sie');
  }
}

/**
 * Get currently selected formality
 */
function getSelectedFormality() {
  const isDu = elements.segmentDu?.classList.contains('active');
  return isDu ? 'du' : 'sie';
}

/**
 * Save settings handler
 */
async function saveSettingsHandler() {
  const selectedAI = getSelectedAI();
  const formality = getSelectedFormality();

  elements.saveSettings.disabled = true;
  showSaveMessage('Speichere...', 'info');

  try {
    // First, check if name needs to be saved
    const nameIsSet = await isNameSet();
    const nameValue = elements.userName ? elements.userName.value.trim() : '';
    
    console.log('[CLX Popup] Save handler - nameIsSet:', nameIsSet, 'nameValue:', nameValue);
    
    // If name is not set but user entered a name, save it first
    if (!nameIsSet && nameValue.length >= 2) {
      console.log('[CLX Popup] Name not set, saving name first...');
      showSaveMessage('Speichere Name...', 'info');
      
      const licenseData = await sendMessage({ action: 'getLicenseData' });
      if (!licenseData || !licenseData.uuid) {
        throw new Error('UUID nicht verfügbar');
      }
      
      console.log('[CLX Popup] Sending name to server:', { uuid: licenseData.uuid, name: nameValue });
      
      const nameResponse = await fetch('https://clx-gmaps-ai-review.celox.io/api/user/name', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          uuid: licenseData.uuid,
          name: nameValue
        })
      });
      
      const nameResult = await nameResponse.json();
      console.log('[CLX Popup] Name save result:', nameResult);
      
      if (!nameResponse.ok) {
        throw new Error(nameResult.error || 'Fehler beim Speichern des Namens');
      }
      
      // Lock the name input
      if (elements.userName) {
        elements.userName.disabled = true;
      }
      if (elements.nameLockIcon) {
        elements.nameLockIcon.style.display = 'block';
      }
      if (elements.saveName) {
        elements.saveName.style.display = 'none';
      }
      if (elements.nameRequiredHint) {
        elements.nameRequiredHint.style.display = 'none';
      }
      
      console.log('[CLX Popup] Name saved successfully');
    } else if (!nameIsSet && nameValue.length < 2) {
      // Name required but not entered
      showSaveMessage('Bitte geben Sie zuerst Ihren Namen ein (min. 2 Zeichen)', 'error');
      if (elements.userName) {
        elements.userName.focus();
      }
      elements.saveSettings.disabled = false;
      return;
    }
    
    // Now save settings
    showSaveMessage('Speichere Einstellungen...', 'info');
    
    const settings = {
      selectedAI,
      formality,
      screenInfo: {
        width: window.screen.width,
        height: window.screen.height,
        colorDepth: window.screen.colorDepth,
        pixelRatio: window.devicePixelRatio
      }
    };

    const response = await sendMessage({ 
      action: 'saveSettings', 
      settings 
    });

    console.log('[CLX Popup] Save response:', response);

    if (response?.success) {
      showSaveMessage('✓ Einstellungen gespeichert!', 'success');
      
      // Reload license data to update status
      await loadLicenseData();
    } else {
      showSaveMessage(response?.error || 'Fehler beim Speichern', 'error');
    }
  } catch (error) {
    console.error('[CLX Popup] Save error:', error);
    showSaveMessage('Fehler: ' + error.message, 'error');
  } finally {
    elements.saveSettings.disabled = false;
  }
}

/**
 * Export logs handler
 */
async function exportLogsHandler() {
  try {
    console.log('[CLX Popup] Exporting logs...');
    const response = await sendMessage({ action: 'exportLogs' });
    
    if (response?.logs) {
      const exportData = {
        exportedAt: new Date().toISOString(),
        totalLogs: response.count,
        logs: response.logs
      };

      const blob = new Blob([JSON.stringify(exportData, null, 2)], { 
        type: 'application/json' 
      });
      const url = URL.createObjectURL(blob);
      
      const a = document.createElement('a');
      a.href = url;
      a.download = `clx-gmaps-logs-${new Date().toISOString().split('T')[0]}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);

      console.log('[CLX Popup] Logs exported successfully');
    }
  } catch (error) {
    console.error('[CLX Popup] Export error:', error);
  }
}

/**
 * Show save message
 */
function showSaveMessage(message, type) {
  if (elements.saveMessage) {
    elements.saveMessage.textContent = message;
    elements.saveMessage.className = 'save-message ' + type;
    
    // Keep info messages visible longer, auto-hide success/error after 4s
    const timeout = type === 'info' ? 0 : 4000;
    if (timeout > 0) {
      setTimeout(() => {
        if (elements.saveMessage.className.includes(type)) {
          elements.saveMessage.className = 'save-message';
        }
      }, timeout);
    }
  }
}

/**
 * Show copied tooltip
 */
function showCopiedTooltip() {
  const tooltip = document.createElement('div');
  tooltip.className = 'copied-tooltip';
  tooltip.textContent = '✓ UUID kopiert!';
  document.body.appendChild(tooltip);
  
  setTimeout(() => {
    tooltip.remove();
  }, 1500);
}

/**
 * Send message to background script
 */
function sendMessage(message) {
  return new Promise((resolve) => {
    try {
      // Check if service worker is available
      if (!chrome.runtime || !chrome.runtime.sendMessage) {
        console.error('[CLX Popup] Chrome runtime not available');
        resolve({ error: 'Chrome runtime not available' });
        return;
      }

      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          console.error('[CLX Popup] Message error:', chrome.runtime.lastError);
          // Retry after a short delay if service worker is not ready
          if (chrome.runtime.lastError.message && 
              (chrome.runtime.lastError.message.includes('message port closed') || 
               chrome.runtime.lastError.message.includes('Extension context invalidated'))) {
            console.log('[CLX Popup] Retrying message after delay...');
            setTimeout(() => {
              chrome.runtime.sendMessage(message, (retryResponse) => {
                if (chrome.runtime.lastError) {
                  resolve({ error: chrome.runtime.lastError.message });
                } else {
                  resolve(retryResponse || { error: 'No response from background' });
                }
              });
            }, 500);
          } else {
            resolve({ error: chrome.runtime.lastError.message });
          }
        } else {
          resolve(response || { error: 'No response from background' });
        }
      });
    } catch (error) {
      console.error('[CLX Popup] sendMessage error:', error);
      resolve({ error: error.message });
    }
  });
}
